# GlobalAmbient.py - Globally set ambient color all actors
#
# Copyright 2005 by Bill Underwood (Underdog)
#
# Based on AdjustAmbient.py by David G. Drumright (ockham)
# Copyright (c) 2005 David G. Drumright (ockham)
#
# IMPORTANT: This code is provided AS-IS with NO WARRANTY. It might damage
#   your data, so be careful and remember that you choose to use this script
#   at your own risk.

from Tkinter import *
import poser, string, types, re, sys, os

def RealBodyPart(act):
  return act.IsBodyPart() and act.Parent() != None

class App:

  def __init__(self, master, textMessage):

    self.scene = poser.Scene()       # hang on to the scene reference
    self.save_all_ambient()          # remember starting ambient values
    self.DirBase = os.path.normpath(os.path.abspath("./Runtime/Python/F3D"))
    try: os.mkdir(self.DirBase)
    except: pass
    (self.First, self.RFlag, self.GFlag, self.BFlag) = (1, 1, 1, 1)

    self.master = master
    self.master.title("Global Ambient")

    self.LevelSlider=Scale(self.master, from_=0.0, to=100.0, orient=HORIZONTAL,
      command=self.LevelSliderChange)
    self.LevelSlider.grid(row=1, column=0)
    self.LevelSlider.set(0.0)

    self.buttonOK = Button(self.master, text="OK",
      command=self.HandleOK, width=5)
    self.buttonOK.grid(row=1, column=1, padx=10)

    self.buttonCancel = Button(self.master, text="Cancel",
      command=self.HandleCancel)
    self.buttonCancel.grid(row=1, column=2, padx=10)

    self.ButtonPanel1 = Frame(self.master, borderwidth=2, relief=FLAT)
    self.ButtonPanel1.grid(row=2, column=0, columnspan=3, pady=5)

    self.buttonRed = Button(self.ButtonPanel1, text="Red", width=7,
      command=self.HandleRed, relief=SUNKEN)
    self.buttonRed.grid(row=0, column=0, padx=2)

    self.buttonGreen = Button(self.ButtonPanel1, text="Green", width=7,
      command=self.HandleGreen, relief=SUNKEN)
    self.buttonGreen.grid(row=0, column=1, padx=2)

    self.buttonBlue = Button(self.ButtonPanel1, text="Blue", width=7,
      command=self.HandleBlue, relief=SUNKEN)
    self.buttonBlue.grid(row=0, column=2, padx=2)

    self.master.protocol('WM_DELETE_WINDOW', self.HandleCancel)

  def restore_ambient(self, act):
    try:
      mats = act.Materials()
      for mat in mats:
        key = "%s/%s" % (act.InternalName(), mat.Name())
        rval, gval, bval = re.split(',', self.OrigAmbience[key])
        mat.SetAmbientColor(float(rval), float(gval), float(bval))
    except: pass

  def save_ambient(self, act):               # save actor's ambient color
    try:                                     # Poser sometimes lies to me
      mats = act.Materials()                 # I can't always figure Poser out
      for mat in mats:                       # loop over materials
        r, g, b = mat.AmbientColor()
        key = "%s/%s" % (act.InternalName(), mat.Name())
        val = "%f,%f,%f" % (r, g, b)
        self.OrigAmbience[key] = val         # save the original ambience
    except: pass                             # ignore errors

  def set_ambient(self, act, pct):           # set actor's ambient color
    try:                                     # Poser sometimes lies to me or
      mats = act.Materials()                 # I can't always figure it out
      for mat in mats:                       # loop over materials
        r, g, b = mat.AmbientColor()         # get ambient tuple
        if self.RFlag: r = pct               # we want to adjust red
        if self.GFlag: g = pct               # we want to adjust green
        if self.BFlag: b = pct               # we want to adjust blue
        mat.SetAmbientColor(r, g, b)         # set ambient tuple
    except: pass                             # ignore errors

  def save_all_ambient(self):                # save all ambient data
    self.OrigAmbience = {}                   # here is where we keep the data
    for fig in self.scene.Figures():         # for each figure..
      for act in fig.Actors():               # ..process just the first actor
        self.save_ambient(act)               # ..saving their material ambience
        break                                # ..then skip the rest of the parts
    for act in self.scene.Actors():          # for all actors in the scene..
      if not RealBodyPart(act):              # ..that aren't really body parts
        self.save_ambient(act)               # ..save their material ambience

  def restore_all_ambient(self):             # restore all saved ambient data
    for act in self.scene.Actors():          # for all actor in the scene
      if RealBodyPart(act):                  # is this actor a body part
        figure = act.ItsFigure()             # if so, get its figure
        if figure == None: continue          # why does this happen??
        for act in figure.Actors():          # pretend we loop over all parts
          self.restore_ambient(act)          # restore ambient 1st body part
          break                              # then exit the loop
      else:                                  # else this is not a body part
        self.restore_ambient(act)            # restore prop's ambient values

  def Update(self):                          # smooth poser event loop a bit
    if self.scene:                           # do we have a scene
      self.scene.ProcessSomeEvents()         # let poser do its stuff
    root.lift()                              # bring Tk window to front
    root.after(100, self.Update)             # call this again soon

  def update_all_ambient(self, sval):
    pct = int(sval) / 100.0                  # Get slider value as a fraction
    for act in self.scene.Actors():          # loop over all actors
      if RealBodyPart(act):                  # if this actor is a body part
        figure = act.ItsFigure()             # get the figure for this actor
        if figure == None: continue          # hmm.. why does this happen?
        for act in figure.Actors():          # pretend this is a loop
          self.set_ambient(act, pct)         # set ambient for first body part
          break                              # surprise! we exit the loop
      else:                                  # else this is not a body part
        self.set_ambient(act, pct)           # set the ambient for this prop
    self.scene.DrawAll()                     # show our changes

  def LevelSliderChange(self, sval):         # the slider has changed values
    if not self.First:                       # the first event is always fake
      self.update_all_ambient(sval)          # otherwise we want to change stuff
    self.First = 0                           # from now on we trust this event

  def HandleRed(self):                       # they clicked the red button
    self.RFlag = not self.RFlag              # toggle the red flag button
    if self.RFlag: self.buttonRed.config(relief=SUNKEN)
    else:          self.buttonRed.config(relief=RAISED)

  def HandleGreen(self):                     # they clicked the green button
    self.GFlag = not self.GFlag              # toggle the green flag button
    if self.GFlag: self.buttonGreen.config(relief=SUNKEN)
    else:          self.buttonGreen.config(relief=RAISED)

  def HandleBlue(self):                      # they clicked the blue button
    self.BFlag = not self.BFlag              # toggle the blue flag button
    if self.BFlag: self.buttonBlue.config(relief=SUNKEN)
    else:          self.buttonBlue.config(relief=RAISED)

  def HandleOK(self):                        # they clicked OK
    self.First = 0                           # we really want this to happen
    val = self.LevelSlider.get()             # get the slider value
    self.LevelSliderChange(val)              # pretend we moved the slider
    self.master.destroy()                    # close Tk window

  def HandleCancel(self):                    # they clicked Cancel
    self.restore_all_ambient()               # recover saved ambient values
    self.master.destroy()                    # close Tk window

root = Tk()
app = App(root, '')
app.Update()
root.mainloop()
